import React, { useState, useEffect } from 'react';
import { setVehicleListData, deleteVehicleAction } from '../../store/reducers/vehicles';
import { setCommonViewData } from '../../store/reducers/common';
import { useDispatch, useSelector } from 'react-redux';
import ModeEditIcon from '@mui/icons-material/ModeEdit';
import DeleteIcon from '@mui/icons-material/Delete';
import ViewIcon from '@mui/icons-material/Visibility';
import { DataGrid } from '@mui/x-data-grid';
import DeleteModal from '../../components/deleteModal/deleteModal';
import { toast } from 'react-toastify';
import EditModal from '../../components/editConfirm/editModal';
import { listVehiclesType } from '../../apiCalls/vehicleType';
import { setVehicleTypeListData } from '../../store/reducers/vehiclesType';
import {
  Avatar,
  AvatarGroup,
  Box,
  Button,
  Grid,
  List,
  ListItemAvatar,
  ListItemButton,
  ListItemSecondaryAction,
  ListItemText,
  MenuItem,
  Stack,
  TextField,
  Typography,
  Tooltip,
  Autocomplete
} from '@mui/material';
import { FormControl, InputAdornment, OutlinedInput } from '@mui/material';
import { PlusIcon } from 'utils/icons';
import ClearAllIcon from '@mui/icons-material/ClearAll';
import { SearchOutlined } from '@ant-design/icons';
import CreateVehicle from './createVehicle';
import { deleteVehicle, listVehicles } from '../../apiCalls/vehicle';
import ViewVehicle from './viewVehicle';
import MainCard from 'components/MainCard';

const Vehicle = () => {
  const dispatch = useDispatch();
  const newVehicleList = useSelector((state) => state?.vehicle?.vehicles) || [];
  const vehicleTypeList = useSelector((state) => state?.vehicleType?.vehiclesType) || [];

  const [open, setOpen] = React.useState(false);
  const [showDeleteModal, setShowDeleteModal] = React.useState(false);
  const [showViewModal, setShowViewModal] = React.useState(false);
  const [showEditModal, setShowEditModal] = React.useState(false);
  const [searchResults, setSearchResults] = useState([]);
  const [searchTerm, setSearchTerm] = useState('');
  const [vehicleType, setVehicleType] = useState('');

  const [list, setList] = React.useState('');
  const [selectedData, setSelectedData] = React.useState(null);
  useEffect(() => {
    listVehiclesType().then((res) => {
      if (res?.data) {
        dispatch(setVehicleTypeListData(res?.data));
      }
    });
  }, []);

  useEffect(() => {
    if (Array.isArray(newVehicleList)) {
      const swappedArray = newVehicleList;
      setList(swappedArray);
    }
  }, [newVehicleList]);

  const search = (text, type) => {
    console.log('type', type);
    return list.filter((vehicle) => {
      const { vehicle_type, plate_number } = vehicle;
      const isVehicleTypeMatch = type ? vehicle_type?.vehicle_type?.toLowerCase() == type.toLowerCase() : true;
      const isPlateNumberMatch = text ? plate_number?.toLowerCase().includes(text.toLowerCase()) : true;
      return isVehicleTypeMatch && isPlateNumberMatch;
    });
  };
  const onSearchClick = (text, type) => {
    setSearchResults(search(text, type));
  };
  const onModeEditIconClick = (e, row) => {
    setSelectedData(row);
    // handleClickOpen(true);
    setShowEditModal(true);
  };
  const onDeleteIconClick = (e, row) => {
    setSelectedData(row);
    setShowDeleteModal(true);
  };
  const onViewIconClick = (e, row) => {
    setSelectedData(row);
    dispatch(setCommonViewData({ row }));
    setShowViewModal(true);
  };

  const columns = [
    {
      field: 'id',
      headerName: 'Sl no',
      sortable: false,
      width: 60,
      disableColumnMenu: true,
      hideSortIcon: true,
      renderCell: (params) => {
        if (list?.length > 0) {
          const indexToUpdate = list?.findIndex((vehicle) => vehicle.id === params.id);
          return <>{indexToUpdate + 1}</>;
        }
      }
    },
    {
      field: 'vehicle_type',
      headerName: 'Vehicle Type',
      width: 170,
      editable: false,
      disableColumnMenu: true,
      sortable: false,
      hideSortIcon: true,
      renderCell: (params) => {
        return <>{params?.row?.vehicle_type?.vehicle_type}</>;
      }
    },
    {
      field: 'plate_number',
      headerName: 'Plate number',
      width: 170,
      sortable: false,
      disableColumnMenu: true,
      editable: false
    },
    {
      field: 'seating_capacity',
      headerName: 'Seat Capacity',
      type: 'number',
      width: 120,
      editable: false,
      disableColumnMenu: true,
      sortable: false,
      hideSortIcon: true,
      renderCell: (params) => {
        return <>{params?.row?.vehicle_type?.seating_capacity}</>;
      }
    },
    {
      field: 'per_km_charge',
      headerName: 'Per km charge',
      type: 'number',
      width: 120,
      editable: false,
      disableColumnMenu: true,
      sortable: false,
      hideSortIcon: true,
      renderCell: (params) => {
        return <>{params?.row?.vehicle_type?.per_km_charge}</>;
      }
    },
    {
      field: 'status',
      headerName: 'Status',
      description: 'This column has a value getter and is not sortable.',
      sortable: false,
      hideMenu: true,
      disableColumnMenu: true,
      width: 170,
      renderCell: (params) => <div className="mx-auto text-center py-1">{params?.row?.status == true ? 'Active' : 'Inactive'} </div>
    },
    {
      field: 'actions',
      headerName: 'Actions',
      sortable: false,
      hideMenu: true,
      disableColumnMenu: true,
      hideSortIcon: true,
      // width: 400,
      renderCell: (params) => {
        return (
          <>
            <Tooltip title="Edit the details" placement="top">
              <ModeEditIcon style={{ cursor: 'pointer' }} color="primary" onClick={(e) => onModeEditIconClick(e, params.row)}>
                Edit
              </ModeEditIcon>
            </Tooltip>{' '}
            &nbsp; &nbsp;
            <Tooltip title="Delete the details" placement="top">
              <DeleteIcon color="primary" style={{ cursor: 'pointer', color: '#ff4d4f' }} onClick={(e) => onDeleteIconClick(e, params.row)}>
                Delete
              </DeleteIcon>
            </Tooltip>{' '}
            &nbsp; &nbsp;
            <Tooltip title="View the details" placement="top">
              <ViewIcon color="primary" style={{ cursor: 'pointer', color: '#1fb141' }} onClick={(e) => onViewIconClick(e, params.row)}>
                View
              </ViewIcon>
            </Tooltip>{' '}
          </>
        );
      }
    }
  ];
  useEffect(() => {
    listVehicles().then((res) => {
      if (res?.data?.status == 1) {
        dispatch(setVehicleListData(res?.data?.vehicles));
      }
    });
  }, []);
  const handleClickOpen = (flag) => {
    setOpen(flag);
  };

  const closeModal = (flag) => {
    setSelectedData(null);
    setOpen(false);
  };
  const closeDeleteModal = (flag) => {
    setSelectedData(null);
    setShowDeleteModal(false);
  };
  const closeViewModal = (flag) => {
    setSelectedData(null);
    setShowViewModal(false);
  };

  const closeEditModal = (flag) => {
    setSelectedData(null);
    setShowEditModal(false);
  };
  const doEdit = (flag) => {
    setShowEditModal(false);
    handleClickOpen(true);
  };
  const deleteItem = async () => {
    try {
      await deleteVehicle(selectedData?.id).then((res) => {
        if (res?.data?.status == 1) {
          toast.success(`${res?.data?.message}`, {
            autoClose: 1500
          });
          dispatch(deleteVehicleAction(selectedData));
          setShowDeleteModal(false);
          setSelectedData(null);
        } else {
          if (res?.response?.data.status == 0) {
            toast.error(`${res?.message}`, {
              autoClose: 1500
            });
          }
        }
      });
    } catch (err) {
      console.log('err', err);
    }
  };
  const doDelete = (flag) => {
    deleteItem(selectedData);
    setShowDeleteModal(false);
  };
  const isOptionEqualToValue = (option, value) => option?.id == value?.id;
  const updateFormValuesWithType = (value) => {
    // console.log('111111111111', value);
    if (value) {
      onSearchClick(searchTerm, value?.vehicle_type);
      setVehicleType(value);
    }
  };
  const onClear = () => {
    setSearchTerm('');
    setVehicleType('');
    setSearchResults([]);
  };

  return (
    <Grid container rowSpacing={4.5} columnSpacing={2.75}>
      <Grid item xs={12} md={12} lg={12}>
        <Grid container alignItems="center" justifyContent="space-between">
          <Grid item>
            <Typography variant="h5">Vehicle Management</Typography>
          </Grid>
          <Grid item>
            <Box sx={{ width: '100%', ml: { xs: 0, md: 1 } }}>
              <FormControl sx={{ width: { xs: '100%', md: 224 } }}>
                <OutlinedInput
                  size="small"
                  id="header-search"
                  startAdornment={
                    <InputAdornment position="start" sx={{ mr: -0.5 }}>
                      <SearchOutlined />
                    </InputAdornment>
                  }
                  onChange={(e) => {
                    setSearchTerm(e?.target?.value);
                    onSearchClick(e?.target?.value, vehicleType?.vehicle_type);
                  }}
                  value={searchTerm}
                  aria-describedby="header-search-text"
                  inputProps={{
                    'aria-label': 'weight'
                  }}
                  placeholder="Search by plate number"
                />
              </FormControl>{' '}
              &nbsp;
              <FormControl sx={{ width: { xs: '100%', md: 224 } }}>
                {vehicleTypeList?.length > 0 && (
                  <Autocomplete
                    sx={{
                      '& .MuiOutlinedInput-root': {
                        padding: '0px',
                        paddingBottom: '0px'
                      },
                      '& .MuiOutlinedInput-root .MuiAutocomplete-input': {
                        paddingLeft: '15px',
                        paddingTop: '-2px'
                      },
                      maxWidth: 420
                    }}
                    options={vehicleTypeList}
                    value={vehicleTypeList.find((vehicle) => vehicle.id == vehicleType.id) || ''}
                    isOptionEqualToValue={(option, value) => isOptionEqualToValue(option, value)}
                    onChange={(e, value) => {
                      console.log('vsl', value);
                      updateFormValuesWithType(value);
                    }}
                    disableClearable={true}
                    getOptionLabel={(option) => option.vehicle_type || 'Select'}
                    renderInput={(params) => (
                      <TextField
                        sx={{
                          '& .MuiInputLabel-root': {
                            height: '1rem'
                          }
                        }}
                        {...params}
                        // InputLabelProps={{ shrink: props?.data != 'add' || formValues.vehicle_type ? true : false }}
                        label="Vehicle Type"
                      />
                    )}
                  />
                )}
              </FormControl>
              &nbsp; &nbsp;
              <Tooltip title="Clear the search values" placement="top">
                <ClearAllIcon
                  color="primary"
                  style={{ marginTop: '10px', cursor: 'pointer', color: '#1677FF' }}
                  onClick={(e) => onClear(e)}
                >
                  Clear
                </ClearAllIcon>
              </Tooltip>{' '}
            </Box>
          </Grid>
          <Grid item>
            <Tooltip title="Create Vehicle" placement="top">
              <Button onClick={() => handleClickOpen(true)} variant="contained" color="primary" startIcon={<PlusIcon />}>
                Create Vehicle
              </Button>
            </Tooltip>
          </Grid>
        </Grid>
        <CreateVehicle handleClickOpenFlag={open} data={selectedData || 'add'} closeModal={(e) => closeModal(e)} />
        <DeleteModal
          handleClickOpenDeleteFlag={showDeleteModal}
          doDelete={(e) => doDelete(e)}
          data={selectedData}
          closeDeleteModal={(e) => closeDeleteModal(e)}
        />
        <EditModal
          handleClickOpenEditFlag={showEditModal}
          doEdit={(e) => doEdit(e)}
          data={selectedData}
          closeEditModal={(e) => closeEditModal(e)}
        />
        <ViewVehicle handleClickOpenViewFlag={showViewModal} data={selectedData} closeViewModal={(e) => closeViewModal(e)} />

        <MainCard sx={{ mt: 2 }} content={false}>
          <div>
            <Box sx={{ height: 400, width: '100%' }}>
              <DataGrid
                sx={{
                  '.MuiDataGrid-columnHeaders': {
                    backgroundColor: '#1677ff',
                    opacity: '.75'
                  },
                  '.MuiDataGrid-columnHeaderTitle': {
                    color: 'white',
                    fontSize: '14px'
                  },
                  '.MuiTablePagination-selectLabel': {
                    marginTop: '12px'
                  },
                  '.MuiTablePagination-displayedRows': {
                    marginTop: '12px'
                  },
                  '.MuiDataGrid-row.Mui-odd': {
                    backgroundColor: '#a5e5a54d'
                  },
                  '.MuiDataGrid-sortIcon': {
                    color: 'white'
                  }
                }}
                rows={searchTerm?.length > 0 || vehicleType?.vehicle_type?.length > 0 ? searchResults : list}
                columns={columns}
                pageSizeOptions={[10, 50, 100]}
                initialState={{
                  pagination: { paginationModel: { pageSize: 10 } }
                }}
                checkboxSelection={false}
                disableRowSelectionOnClick
                getRowClassName={(params) => (params.indexRelativeToCurrentPage % 2 === 0 ? 'Mui-even' : 'Mui-odd')}
              />
            </Box>
          </div>
        </MainCard>
      </Grid>
    </Grid>
  );
};

export default Vehicle;
